<?php

namespace App\Sys\Services\Client;

use App\Sys\Services;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Validator;
use App\Sys\Repositories\Client\ClientSettingRepository;
use App\Sys\Repositories\Client\ClientRepository;
use Exception;

class ClientSettingService extends Services
{
    protected $clientSettingRepository;
    protected $clientRepository;

    public function __construct(
        ClientSettingRepository $clientSettingRepository,
        ClientRepository $clientRepository
    ) {
        $this->clientSettingRepository = $clientSettingRepository;
        $this->clientRepository = $clientRepository;
    }

    public function registerMachine($request)
    {
        $rules = [
            'otp' => ['required', 'numeric', 'integer', 'digits:6'],
            'email' => ['required', 'email', 'max:50'],
            'common_name' => ['required', 'string', 'max:255'],
            'organizational_unit_name' => ['required', 'string', 'max:255'],
            'organization_name' => ['required', 'string', 'max:255'],
            'tax_number' => ['required', 'string', 'digits:15'],
            'business_category' => ['required', 'string', 'max:50'],
            'short_ad_category' => ['required', 'string', 'max:50'],
            'egs_serial_number' => ['required', 'string', 'max:255', 'unique:client_settings,egs_serial_number'],
            'registration_number' => ['required', 'string', 'max:50'],
            'client_id' => ['nullable', 'integer', 'exists:clients,id'],
        ];

        $validator = Validator::make($request, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());
            return false;
        }

        try {
            $settings = new \Bl\FatooraZatca\Objects\Setting(
                $request['otp'],
                $request['email'],
                $request['common_name'],
                $request['organizational_unit_name'],
                $request['organization_name'],
                $request['tax_number'],
                $request['short_ad_category'],
                $request['business_category'],
                $request['egs_serial_number'],
                $request['registration_number']
            );

            $result = \Bl\FatooraZatca\Zatca::generateZatcaSetting($settings);
            
            // Handle client creation or linking
            if (isset($request['client_id']) && $request['client_id']) {
                $client = $this->clientRepository->findByClientId($request['client_id']);
                $clientId = $client->id;
            } else {
                $clientData = [
                    'name' => $request['organization_name'],
                    'email' => $request['email'],
                    'tax_number' => $request['tax_number'],
                ];

                $client = $this->clientRepository->create($clientData);
                $clientId = $client->id;
            }

            $clientSettingData = [
                'client_id' => $clientId,
                'machine_id' => Str::random(32),
                'email' => $request['email'],
                'common_name' => $request['common_name'],
                'organizational_unit_name' => $request['organizational_unit_name'],
                'organization_name' => $request['organization_name'],
                'tax_number' => $request['tax_number'],
                'short_ad_category' => $request['short_ad_category'],
                'egs_serial_number' => $request['egs_serial_number'],
                'business_category' => $request['business_category'],
                'registration_number' => $request['registration_number'],
                'result' => $result,
            ];

            $clientSetting = $this->clientSettingRepository->create($clientSettingData);

            return $clientSetting->machine_id;
        } catch (Exception $e) {
            $errorMessage = $e->getMessage();
            $errorData = json_decode($errorMessage, true);
            if (isset($errorData['errors'])) {
                $this->setError([['tax_number' => $errorData['errors']]]);
            } else {
                $this->setError([['general' => 'An error occurred while registering the client.']]);
            }
            return false;
        }
    }
}
