<?php

namespace App\Sys\Services\Invoice;

use App\Enums\InvoiceStatus;
use Exception;
use App\Sys\Services;
use Illuminate\Support\Str;
use App\Rules\InvoiceValidationRule;
use Illuminate\Support\Facades\Validator;
use Bl\FatooraZatca\Classes\TaxCategoryCode;
use App\Sys\Repositories\Invoice\InvoiceRepository;
use App\Sys\Repositories\Client\ClientSettingRepository;

class InvoiceService extends Services
{
    protected $clientSettingRepository;
    protected $invoiceRepository;

    public function __construct(ClientSettingRepository $clientSettingRepository, InvoiceRepository $invoiceRepository)
    {
        $this->clientSettingRepository = $clientSettingRepository;
        $this->invoiceRepository = $invoiceRepository;
    }

    public function reportInvoice($request)
    {
        $validator = Validator::make($request, InvoiceValidationRule::rules());

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $clientSetting = $this->clientSettingRepository->findByMachineId($request['machine_id']);

        $lastInvoice = $this->invoiceRepository->lastClientInvoice($request['machine_id'], $request['invoice_type']);

        $invoiceNumber = $this->generateInvoiceNumber($request['machine_id'], $request['invoice_type']);

        $generatedUuid = Str::uuid();

        // Store invoice initially with PENDING status
        $invoice = $this->invoiceRepository->createInvoice($request, $invoiceNumber, $generatedUuid);


        $report = $this->processZatcaInvoice($invoice, $request, $clientSetting, $lastInvoice);
        $result = [
            'status' => $report->status,
            'invoice_number' => $report->invoice_number,
            'machine_id' => $report->machine_id,
            'invoice_kind' => $report->invoice_kind
        ];
        return $result;
    }

    protected function processZatcaInvoice($invoice, $request, $clientSetting, $lastInvoice = null)
    {
        try {
            // Prepare seller and invoice objects
            $seller = new \Bl\FatooraZatca\Objects\Seller(
                $clientSetting->registration_number,
                $request['street_name'],
                $request['building_number'],
                $request['plot_identification'],
                $request['city_sub_division'],
                $request['city'],
                $request['postal_number'],
                $clientSetting->tax_number,
                $clientSetting->organization_name,
                $clientSetting->result['private_key'],
                $clientSetting->result['cert_production'],
                $clientSetting->result['secret_production']
            );

            $invoiceType = config('invoiceTypes.invoice_types')[$request['invoice_type']];
            $paymentType = config('invoiceTypes.payment_types')[$request['payment_type']];

            $invoiceItems = [];
            foreach ($request['invoice_items'] as $index => $item) {
                $taxCategoryCode = isset($item['tax_category'])
                    ? config('invoiceTypes.tax_category_codes')[$item['tax_category']]
                    : TaxCategoryCode::STANDARD_RATE;

                $invoiceItems[] = new \Bl\FatooraZatca\Objects\InvoiceItem(
                    $index + 1,
                    $item['name'],
                    $item['quantity'],
                    $item['unit_price'],
                    $item['discount'] ?? 0,
                    $item['tax_amount'],
                    $item['tax_rate'],
                    $item['total_amount'],
                    $item['discount_note'] ?? null,
                    $taxCategoryCode
                );
            }

            $fatooraInvoice = new \Bl\FatooraZatca\Objects\Invoice(
                $invoice->id,
                $invoice->invoice_number,
                $invoice->uuid,
                $invoice->invoice_date,
                $invoice->invoice_time,
                $invoiceType,
                $paymentType,
                $request['total_amount_without_tax'],
                [],
                $request['total_tax'],
                $request['total_amount_with_tax'],
                $invoiceItems,
                null,
                null,
                $lastInvoice->invoice_hash ?? null,
                $request['payment_note'],
                'SAR',
                $request['vat_rate'],
                null,
                $request['rounding_amount'] ?? 0
            );

            // Process B2B or B2C invoice
            $isB2B = isset($request['buyer_registration_name']);

            if ($isB2B) {
                $client = new \Bl\FatooraZatca\Objects\Client(
                    $request['buyer_registration_name'],
                    $request['buyer_tax_number'],
                    $request['buyer_postal_number'],
                    $request['buyer_street_name'],
                    $request['buyer_building_number'],
                    $request['buyer_plot_identification'],
                    $request['buyer_city_subdivision_name'],
                    $request['buyer_city']
                );
                $report = \Bl\FatooraZatca\Invoices\B2B::make($seller, $fatooraInvoice, $client)->report();
            } else {
                $report = \Bl\FatooraZatca\Invoices\B2C::make($seller, $fatooraInvoice)->report();
            }

            // Save QR code image
            $qrImage = $report->getQrImage();
            $folderPath = public_path("qr_codes/{$clientSetting->client_id}/{$clientSetting->machine_id}");

            // Create directory if it doesn't exist
            if (!file_exists($folderPath)) {
                mkdir($folderPath, 0755, true);
            }

            // Save image with invoice invoice number as filename
            $imagePath = "{$folderPath}/{$invoice->invoice_number}.svg";
            file_put_contents($imagePath, $qrImage);

            // Update invoice on success
            $this->invoiceRepository->updateInvoiceStatus($invoice, 'SUCCESS', $report->getInvoiceHash(), $report);

            return $invoice;
        } catch (Exception $e) {
            $this->invoiceRepository->updateInvoiceStatus($invoice, 'FAILED', null, null);
            $this->setError($e->getMessage());
            return false;
        }
    }

    public function machineInvoices($machineId)
    {
        $result = $this->invoiceRepository->getMachineInvoices($machineId);

        if (!$result) {
            return false;
        }

        return $result;
    }

    public function resendInvoice($invoiceId)
    {
        $invoice = $this->invoiceRepository->findFailedInvoice($invoiceId);

        if (!$invoice) {
            $this->setError(trans('messages.Invoice not found or status is not FAILED or PENDING'));
            return false;
        }

        // Prepare request data from existing invoice
        $request = [
            'machine_id' => $invoice->machine_id,
            'invoice_type' => $invoice->invoice_type,
            'payment_type' => $invoice->payment_type,
            'vat_rate' => $invoice->vat_rate,
            'total_amount_without_tax' => $invoice->total_amount_without_tax,
            'total_tax' => $invoice->total_tax,
            'total_amount_with_tax' => $invoice->total_amount_with_tax,
            'rounding_amount' => $invoice->rounding_amount,
            'street_name' => $invoice->seller_street_name,
            'building_number' => $invoice->seller_building_number,
            'plot_identification' => $invoice->seller_plot_identification,
            'city_sub_division' => $invoice->seller_city_sub_division,
            'city' => $invoice->seller_city,
            'postal_number' => $invoice->seller_postal_number,
            'buyer_registration_name' => $invoice->buyer_registration_name,
            'buyer_tax_number' => $invoice->buyer_tax_number,
            'buyer_street_name' => $invoice->buyer_street_name,
            'buyer_building_number' => $invoice->buyer_building_number,
            'buyer_plot_identification' => $invoice->buyer_plot_identification,
            'buyer_city_subdivision_name' => $invoice->buyer_city_subdivision_name,
            'buyer_city' => $invoice->buyer_city,
            'buyer_postal_number' => $invoice->buyer_postal_number,
            'payment_note' => $invoice->payment_note,
            'parent_invoice_id' => $invoice->parent_invoice_id,
            'invoice_note' => $invoice->invoice_note,
            'invoice_items' => $invoice->invoice_items,
        ];

        $newInvoiceNumber = $this->generateInvoiceNumber($invoice->machine_id, $invoice->invoice_type);

        // Reset invoice status to PENDING
        $this->invoiceRepository->updateInvoiceStatus($invoice, InvoiceStatus::PENDING, null, null, $newInvoiceNumber);


        $clientSetting = $this->clientSettingRepository->findByMachineId($invoice->machine_id);

        $lastInvoice = $this->invoiceRepository->lastClientInvoice($invoice->machine_id, $invoice->invoice_type);

        $report = $this->processZatcaInvoice($invoice, $request, $clientSetting, $lastInvoice);

        $result = [
            'status' => $report->status,
            'invoice_number' => $report->invoice_number,
            'machine_id' => $report->machine_id,
            'invoice_kind' => $report->invoice_kind
        ];
        return $result;
    }

    protected function generateInvoiceNumber($machineId, $invoiceType)
    {
        $prefixMap = [
            'TAX_INVOICE' => 'INV',
            'CREDIT_NOTE' => 'INVC',
            'DEBIT_NOTE' => 'INVD',
        ];
        $prefix = $prefixMap[$invoiceType] ?? 'INV';

        $lastInvoice = $this->invoiceRepository->lastClientInvoice($machineId, $invoiceType);

        $nextNumber = ($lastInvoice && preg_match("/$prefix(\d+)/", $lastInvoice->invoice_number, $matches))
            ? intval($matches[1]) + 1
            : 1;
        $result = $prefix . str_pad($nextNumber, 3, '0', STR_PAD_LEFT);
        return $result;
    }
}
